/************************************************************************
 *
 * \file: util_mutex_robust.c
 *
 * <brief description>.
 * Implements the robust variant of the pthread_mutex in a simplified way
 * <detailed description>
 * \component: utility - util_mutex_robust
 *
 * \author: Dhanasekaran D / RBEI/ECF3 / dhanasekaran.d(o)in.bosch.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <asm-generic/errno.h>
#include "util_mutex_robust.h"

/* Initialize pthread mutex with robust and process_shared attribute */
int UTIL_mutex_robust_init(util_mutex_robust *mutex,
							mutex_robust_recovery_handler hdlr)
{
	int rc;
	pthread_mutexattr_t attr;

	pthread_mutexattr_init(&attr);
	pthread_mutexattr_setpshared(&attr, PTHREAD_PROCESS_SHARED);
	pthread_mutexattr_setrobust(&attr, PTHREAD_MUTEX_ROBUST);

	rc = pthread_mutex_init(&mutex->m, &attr);

	pthread_mutexattr_destroy(&attr);

	mutex->section = 0;
	mutex->recovery_handler = hdlr;

	return rc;
}

/* PRQA: Lint Message 454: Unlock done in UTIL_mutex_robust_unlock function*/
/* PRQA: Lint Message 456: Mutex recovery failed treated as not acquired*/
/*lint -save -e454 -e456 */

/* Acquires the mutex lock*/
int UTIL_mutex_robust_lock(util_mutex_robust *mutex)
{
	int rc;
	rc = pthread_mutex_lock(&mutex->m);

	if (EOWNERDEAD == rc) {
		if (mutex->recovery_handler != NULL) {
			if (mutex->recovery_handler(mutex) == 0) {
				pthread_mutex_consistent(&mutex->m);
				rc = UTIL_MUTEX_RECOVERED;
			} else {
				pthread_mutex_unlock(&mutex->m);
				rc = UTIL_MUTEX_RECOVERY_FAILED;
			}
		} else { /*without recovery handler*/
			pthread_mutex_consistent(&mutex->m);
			rc = UTIL_MUTEX_RECOVERED_NO_HANDLER;
		}
	}

	return rc;
}
/*lint -restore*/

/* Releases the mutex lock */
int UTIL_mutex_robust_unlock(util_mutex_robust *mutex)
{
	/* PRQA: Lint Message 455: Lock done in UTIL_mutex_robust_lock function */
	return pthread_mutex_unlock(&mutex->m);/*lint !e455 */
}

/* Destroys the pthread mutex */
int UTIL_mutex_robust_destroy(util_mutex_robust *mutex)
{
	return pthread_mutex_destroy(&mutex->m);
}
